import {
    Controller,
    Get,
    Post,
    Put,
    Param,
    Body,
    UseInterceptors,
    UploadedFile,
    ParseIntPipe,
    Delete,
    Query,
    UseGuards,
    Req,
} from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { BlogService } from './blog.service';
import { CreateBlogDto } from './dto/create-blog.dto';
import { UpdateBlogDto } from './dto/update-blog.dto';
import { multerConfig } from 'src/common/utils/multer.config';
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';

@Controller('blogs')
export class BlogController {
    constructor(private readonly blogService: BlogService) { }

    @Post('create')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads')))
    async create(@Body() dto: CreateBlogDto, @UploadedFile() image: Express.Multer.File, @Req() req: any) {
        const admin = req.user as any;
        const createdBy = admin?.id;
        return await this.blogService.create(dto, image, createdBy);
    }

    @Get('list')
    async findAll(
        @Query('status') status?: number,
        @Query('page') page: number = 1,
        @Query('limit') limit: number = 10,
        @Query('search') search?: string,
    ) {
        const parsedStatus = status !== undefined ? Number(status) : undefined;
        return await this.blogService.findAll(parsedStatus, Number(page), Number(limit), search);
    }

    @Get('findby/:id')
    async findOne(@Param('id', ParseIntPipe) id: number) {
        return await this.blogService.findOne(id);
    }

    @Get('findbyslug/:slug')
    async findBySlug(@Param('slug') slug: string) {
        return await this.blogService.findBySlug(slug);
    }


    @Put('updateby/:id')
    @UseGuards(AdminJwtAuthGuard)
    @UseInterceptors(FileInterceptor('image', multerConfig('uploads')))
    async update(
        @Param('id', ParseIntPipe) id: number,
        @Body() dto: UpdateBlogDto,
        @UploadedFile() image: Express.Multer.File,
        @Req() req: any
    ) {
        const admin = req.user as any;
        const updatedBy = admin?.id;
        return await this.blogService.update(id, dto, image, updatedBy);
    }


    @Put('toggleStatus/:id')
    @UseGuards(AdminJwtAuthGuard)
    async toggleStatus(@Param('id', ParseIntPipe) id: number) {
        return await this.blogService.toggleStatus(id);
    }
}
