import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Not } from 'typeorm';
import { Category } from './category.entity';
import { CreateCategoryDto } from './dto/create-category.dto';
import { UpdateCategoryDto } from './dto/update-category.dto';

@Injectable()
export class CategoryService {
    constructor(
        @InjectRepository(Category)
        private readonly categoryRepo: Repository<Category>,
    ) { }

    async create(dto: CreateCategoryDto, createdBy: number) {
        try {
            const category = this.categoryRepo.create({
                ...dto,
                created_by: createdBy,
            });
            await this.categoryRepo.save(category);
            return { status: true, message: 'Category created successfully', data: category };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    async findAll() {
        try {
            const data = await this.categoryRepo.find({
                where: { status: Not(2) },
                order: { id: 'DESC' },
            });
            return { status: true, message: 'List of category found', data };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    async findById(id: number) {
        try {
            const data = await this.categoryRepo.findOne({
                where: { id, status: Not(2) },
            });
            if (!data) return { status: false, message: 'Category not found' };
            return { status: true, message: 'Category found', data };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    async update(id: number, dto: UpdateCategoryDto, updatedBy: number) {
        try {
            const category = await this.categoryRepo.findOne({ where: { id, status: 1 } });
            if (!category) return { status: false, message: 'Category not found' };

            Object.assign(category, dto, { updated_by: updatedBy });
            await this.categoryRepo.save(category);
            return { status: true, message: 'Category updated successfully', data: category };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    async toggleStatus(id: number) {
        try {
            const category = await this.categoryRepo.findOne({ where: { id } });
            if (!category) return { status: false, message: 'Category not found' };

            category.status = category.status === 1 ? 0 : 1;
            await this.categoryRepo.save(category);

            return {
                status: true,
                message: `Category status changed to ${category.status === 1 ? 'Active' : 'Inactive'}`,
                data: category,
            };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    async softDelete(id: number) {
        try {
            const category = await this.categoryRepo.findOne({ where: { id } });
            if (!category) return { status: false, message: 'Category not found' };

            category.status = 2; // 2 = soft deleted
            await this.categoryRepo.save(category);
            return { status: true, message: 'Category soft deleted successfully' };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }
}
