import { Injectable, Logger } from '@nestjs/common';
import * as nodemailer from 'nodemailer';

@Injectable()
export class MailService {
  private transporter: nodemailer.Transporter;
  private readonly logger = new Logger(MailService.name);
  private mainProjectEmail: string;

  constructor() {
    const host = process.env.MAIL_HOST || 'smtp.gmail.com';
    const port = Number(process.env.MAIL_PORT || 587);
    const user = process.env.MAIL_USER;
    const pass = process.env.MAIL_PASS;
    this.mainProjectEmail =
      process.env.MAIN_PROJECT_EMAIL || 'wisamahmed851@gmail.com';
    this.transporter = nodemailer.createTransport({
      host,
      port,
      secure: process.env.MAIL_SECURE === 'true' || false,
      auth: {
        user,
        pass,
      },
    });
  }

  /**
   * Generic send mail method
   * @param from - sender email (string) or formatted "Name <email>"
   * @param to - recipients (string or string[])
   * @param subject - subject string
   * @param text - plain text body (optional)
   * @param html - html body (optional)
   */
  async sendMail({
    from,
    to,
    subject,
    text,
    html,
  }: {
    from: string;
    to: string | string[];
    subject: string;
    text?: string;
    html?: string;
  }) {
    const mailOptions: nodemailer.SendMailOptions = {
      from,
      to,
      subject,
      text,
      html,
    };

    try {
      const info = await this.transporter.sendMail(mailOptions);
      this.logger.log(`Email sent: ${info.messageId}`);
      return { success: true, message: 'Email sent', info };
    } catch (error) {
      this.logger.error('Error sending email', error as any);
      return { success: false, message: 'Failed to send email', error };
    }
  }

  /**
   * Helper to send contact-us email to project main email.
   * Accepts contact form fields and formats an HTML message.
   */
  async sendContactUs({
    name,
    email,
    subject,
    message,
  }: {
    name: string;
    email: string;
    subject: string;
    message: string;
  }) {
    const from = `${name} <${email}>`;
    const to = this.mainProjectEmail;

    const html = `
      <div style="font-family: Arial, sans-serif; padding: 18px;">
        <h2>📩 New Contact Us Message</h2>
        <p><strong>Name:</strong> ${this.escapeHtml(name)}</p>
        <p><strong>Email:</strong> ${this.escapeHtml(email)}</p>
        <p><strong>Subject:</strong> ${this.escapeHtml(subject)}</p>
        <hr />
        <p><strong>Message:</strong></p>
        <div style="background:#f7f7f7; padding:12px; border-radius:6px;">${this.escapeHtml(message).replace(/\n/g, '<br/>')}</div>
      </div>
    `;

    return this.sendMail({
      from,
      to,
      subject: `Contact Us: ${subject}`,
      html,
      text: message,
    });
  }

  // Minimal HTML-escaping to avoid injection in mail body
  private escapeHtml(str: string) {
    if (!str) return '';
    return str
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }
}
