import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ServiceDetailImage } from './service-detail-image.entity';
import * as fs from 'fs';
import * as path from 'path';

@Injectable()
export class ServiceDetailImageService {
    constructor(
        @InjectRepository(ServiceDetailImage)
        private readonly imageRepo: Repository<ServiceDetailImage>,
    ) { }

    // 📜 Get all images (optional: filter by detail ID)
    async findAll(detailId?: number) {
        try {
            const where = detailId ? { service_detail_id: detailId } : {};
            const images = await this.imageRepo.find({ where });

            return {
                status: true,
                message: 'Images fetched successfully',
                data: images,
            };
        } catch (error) {
            return { status: false, message: error.message, data: null };
        }
    }

    // 🔁 Toggle image status (1 ⇄ 0)
    async toggleStatus(id: number) {
        try {
            const image = await this.imageRepo.findOne({ where: { id } });

            if (!image) {
                return { status: false, message: `Image with ID ${id} not found` };
            }

            image.status = image.status === 1 ? 0 : 1;
            await this.imageRepo.save(image);

            return {
                status: true,
                message: `Image status updated to ${image.status === 1 ? 'Active' : 'Inactive'}`,
                data: image,
            };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }

    // ❌ Delete image (from DB and server)
    async deleteImage(id: number) {
        try {
            const image = await this.imageRepo.findOne({ where: { id } });

            if (!image) {
                return { status: false, message: `Image with ID ${id} not found` };
            }

            // Delete file from server
            const imagePath = path.join(process.cwd(), image.image_path);
            if (fs.existsSync(imagePath)) {
                fs.unlinkSync(imagePath);
            }

            // Remove record from DB
            await this.imageRepo.remove(image);

            return {
                status: true,
                message: 'Image deleted successfully (DB + Server)',
            };
        } catch (error) {
            return { status: false, message: error.message };
        }
    }
}
