import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Like } from 'typeorm';
import { ServiceDetail } from './service-detail.entity';
import { CreateServiceDetailDto } from './dto/create-service-detail.dto';
import { UpdateServiceDetailDto } from './dto/update-service-detail.dto';
import { Service } from 'src/service/service.entity';

@Injectable()
export class ServiceDetailService {
  constructor(
    @InjectRepository(ServiceDetail)
    private readonly repo: Repository<ServiceDetail>,
    @InjectRepository(Service)
    private readonly serviceRepo: Repository<Service>,
  ) {}

  async create(dto: CreateServiceDetailDto, createdBy?: number) {
    try {
      const service = await this.serviceRepo.findOne({
        where: { id: dto.service_id, status: 1 },
      });

      if (!service) {
        return {
          status: false,
          message: `Service with ID ${dto.service_id} not found`,
          data: null,
        };
      }

      const detail = this.repo.create({ ...dto, created_by: createdBy });
      const saved = await this.repo.save(detail);
      return {
        status: true,
        message: 'Service detail created successfully',
        data: saved,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to create service detail',
        data: error.message,
      };
    }
  }

  // Get All Service Details (with optional filters)
  async findAll(
    status?: number,
    page: number = 1,
    limit: number = 10,
    search?: string,
  ) {
    try {
      const skip = (page - 1) * limit;

      const whereConditions: any = {};
      if (status !== undefined) whereConditions.status = status;
      if (search) whereConditions.title = Like(`%${search}%`);

      const [data, total] = await this.repo.findAndCount({
        where: whereConditions,
        relations: ['service'],
        skip,
        take: limit,
        order: { id: 'DESC' },
      });

      return {
        status: true,
        message: 'Service details fetched successfully',
        data,
        pagination: {
          total,
          currentPage: page,
          totalPages: Math.ceil(total / limit),
          limit,
        },
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch service details',
        data: error.message,
      };
    }
  }

  async findOne(id: number) {
    try {
      const detail = await this.repo.findOne({
        where: { id },
        relations: ['service'],
      });

      if (!detail) {
        return {
          status: false,
          message: `Service detail with ID ${id} not found`,
          data: null,
        };
      }

      return {
        status: true,
        message: 'Service detail fetched successfully',
        data: detail,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch service detail',
        data: error.message,
      };
    }
  }

  async update(id: number, dto: UpdateServiceDetailDto, updatedBy?: number) {
    try {
      const existing = await this.repo.findOne({ where: { id } });
      if (!existing) {
        return {
          status: false,
          message: `Service detail with ID ${id} not found`,
          data: null,
        };
      }

      Object.assign(existing, dto, { updated_by: updatedBy });
      const updated = await this.repo.save(existing);
      return {
        status: true,
        message: 'Service detail updated successfully',
        data: updated,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to update service detail',
        data: error.message,
      };
    }
  }

  async toggleStatus(id: number) {
    try {
      const detail = await this.repo.findOne({ where: { id } });
      if (!detail) {
        return {
          status: false,
          message: `Service detail with ID ${id} not found`,
          data: null,
        };
      }

      detail.status = detail.status === 1 ? 0 : 1;
      const updated = await this.repo.save(detail);

      return {
        status: true,
        message: `Service detail ${
          updated.status === 1 ? 'activated' : 'deactivated'
        } successfully`,
        data: updated,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to toggle service detail status',
        data: error.message,
      };
    }
  }
}
