import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Param,
  Body,
  Query,
  UploadedFiles,
  UseGuards,
  Req,
  UseInterceptors,
  ParseIntPipe,
} from '@nestjs/common';
import { ServiceService } from './service.service';
import {
  AnyFilesInterceptor,
  FilesInterceptor,
} from '@nestjs/platform-express';
import { multerConfig } from 'src/common/utils/multer.config';
import { CreateServiceDto } from './dto/create-service.dto';
import { UpdateServiceDto } from './dto/update-service.dto';
import { AdminJwtAuthGuard } from 'src/auth/admin/admin-jwt.guard';
import { inspect } from 'util';

@Controller('services')
export class ServiceController {
  constructor(private readonly serviceService: ServiceService) {}

  @Post('create')
  @UseGuards(AdminJwtAuthGuard)
  @UseInterceptors(AnyFilesInterceptor(multerConfig('uploads/services')))
  async create(
    @UploadedFiles() files: Express.Multer.File[],
    @Body() dto: any,
    @Req() req: any,
  ) {
    console.log(dto);
    const admin = req.user as any;
    const createdBy = admin?.id;

    // Step 1: Parse details JSON if needed
    if (dto.details && typeof dto.details === 'string') {
      try {
        dto.details = JSON.parse(dto.details);
      } catch {
        return { message: 'Invalid JSON format for details' };
      }
    }

    // Step 2: Separate file categories
    const mainImages = files.filter((f) => f.fieldname === 'images');
    const serviceImageFile = files.find(
      (f) => f.fieldname === 'service_image_link',
    );
    const groupFiles = files.filter((f) =>
      f.fieldname.startsWith('imageGroups['),
    );

    // Step 3: Attach detail images
    if (dto.details && mainImages.length > 0) {
      dto.details = dto.details.map((detail, index) => ({
        ...detail,
        image: mainImages[index]?.filename || null,
      }));
    }

    // Step 4: Attach image groups
    (dto as any).imageGroups = (dto as any).imageGroups || [];
    files.forEach((file) => {
      const match = file.fieldname.match(/imageGroups\[(\d+)\]/);
      if (match) {
        const index = parseInt(match[1], 10);
        (dto as any).imageGroups[index] = (dto as any).imageGroups[index] || [];
        (dto as any).imageGroups[index].push(file.filename);
      }
    });

    if (Array.isArray(dto.details) && Array.isArray((dto as any).imageGroups)) {
      const imageGroups = (dto as any).imageGroups as string[][];
      dto.details = dto.details.map((detail, index) => ({
        ...detail,
        images:
          imageGroups[index]?.map((filename: string) => ({
            image_path: filename,
            status: 1,
          })) || [],
      }));
    }

    // ✅ Step 5: Match bullet icons with pattern: details[i][bullets][j][bulletIcons]
    dto.details = dto.details?.map((detail, detailIndex) => {
      if (detail.bullets && detail.bullets.length > 0) {
        detail.bullets = detail.bullets.map((bullet, bulletIndex) => {
          const matchingFile = files.find(
            (f) =>
              f.fieldname ===
              `details[${detailIndex}][bullets][${bulletIndex}][bulletIcons]`,
          );

          return {
            title: bullet.title,
            text: bullet.text || '',
            alt_icon: bullet.alt_icon || '',
            icon: matchingFile ? `services/${matchingFile.filename}` : null,
          };
        });
      }
      return detail;
    });

    // Step 6: Call service
    return this.serviceService.create(
      dto,
      createdBy,
      serviceImageFile?.filename,
    );
  }

  @Get('list')
  findAll(
    @Query('limit') limit?: number,
    @Query('offset') offset?: number,
    @Query('status') status?: number,
  ) {
    return this.serviceService.findAll(status, limit, offset);
  }

  @Get('findby/:slug')
  findById(@Param('slug') slug: string) {
    return this.serviceService.findById(slug);
  }

  @Put('updateby/:id')
  @UseGuards(AdminJwtAuthGuard)
  @UseInterceptors(AnyFilesInterceptor(multerConfig('uploads/services')))
  async update(
    @Param('id') id: number,
    @UploadedFiles() files: Express.Multer.File[],
    @Body() dto: any,
    @Req() req: any,
  ) {
    console.log(dto);
    const admin = req.user as any;
    const updatedBy = admin?.id;

    // ✅ Step 1: Parse details if JSON string
    if (dto.details && typeof dto.details === 'string') {
      try {
        dto.details = JSON.parse(dto.details);
      } catch {
        return { message: 'Invalid JSON format for details' };
      }
    }

    // ✅ Step 2: Extract files
    const mainImages = files.filter((f) => f.fieldname === 'images');
    const serviceImageFile = files.find(
      (f) => f.fieldname === 'service_image_link',
    );
    const groupFiles = files.filter((f) =>
      f.fieldname.startsWith('imageGroups['),
    );

    // ✅ Step 3: Attach main images (for each detail)
    if (dto.details && mainImages.length > 0) {
      dto.details = dto.details.map((detail, index) => ({
        ...detail,
        image: mainImages[index]?.filename || detail.image || null,
      }));
    }

    // ✅ Step 4: Handle grouped detail images properly
    (dto as any).imageGroups = (dto as any).imageGroups || [];

    files.forEach((file) => {
      const match = file.fieldname.match(/imageGroups\[(\d+)\]/);
      if (match) {
        const index = parseInt(match[1], 10);
        (dto as any).imageGroups[index] = (dto as any).imageGroups[index] || [];
        (dto as any).imageGroups[index].push({
          image_path: file.filename,
          status: 1,
        });
      }
    });

    // ✅ Step 5: Attach imageGroups to corresponding details
    if (Array.isArray(dto.details) && Array.isArray((dto as any).imageGroups)) {
      const imageGroups = (dto as any).imageGroups as any[][];
      dto.details = dto.details.map((detail, index) => ({
        ...detail,
        images: imageGroups[index] || detail.images || [],
      }));
    }

    // ❌ Removed this duplicate block that caused `[object Object]`:
    // (you had a second `if (Array.isArray(dto.details) && Array.isArray(imageGroups))` section)
    // It treated imageGroups as strings and rewrapped objects — leading to "services/[object Object]"

    // ✅ Step 6: Handle bullet icons
    dto.details = dto.details?.map((detail, detailIndex) => {
      if (detail.bullets && detail.bullets.length > 0) {
        detail.bullets = detail.bullets.map((bullet, bulletIndex) => {
          const matchingFile = files.find(
            (f) =>
              f.fieldname ===
              `details[${detailIndex}][bullets][${bulletIndex}][bulletIcons]`,
          );

          let iconValue = bullet.icon;
          if (matchingFile) {
            iconValue = `services/${matchingFile.filename}`;
          } else if (bullet.icon === undefined) {
            iconValue = bullet.existing_icon || bullet.icon;
          }

          return {
            id: bullet.id,
            title: bullet.title,
            text: bullet.text || '',
            alt_icon: bullet.alt_icon || '',
            icon: iconValue,
            status: bullet.status ?? 1,
          };
        });
      }
      return detail;
    });

    // ✅ Step 7: Call the service update
    return this.serviceService.update(
      +id,
      dto,
      updatedBy,
      serviceImageFile?.filename,
    );
  }

  @Put('toggleStatus/:id')
  toggle(@Param('id', ParseIntPipe) id: number) {
    return this.serviceService.toggleStatus(id);
  }

  @Post('create-page-details')
  Pagecreate(@Body() body: any) {
    return this.serviceService.createPageHeading(body);
  }

  @Get('get-page-details/:type')
  PagefindOne(@Param('type') typer: string) {
    return this.serviceService.findPageOne(typer);
  }

  @Put('update-page-details/:id')
  Pageupdate(@Param('id') id: number, @Body() body: any) {
    return this.serviceService.updatePage(id, body);
  }

  @Delete('delete-page/:id')
  Pageremove(@Param('id') id: number) {
    return this.serviceService.remove(id);
  }

  @Delete('deleteby/:id')
  @UseGuards(AdminJwtAuthGuard)
  async softDelete(@Param('id') id: number) {
    return await this.serviceService.softDelete(id);
  }
}
