import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Like, DataSource, In, Not } from 'typeorm';
import { Service } from './service.entity';
import { CreateServiceDto } from './dto/create-service.dto';
import { UpdateServiceDto } from './dto/update-service.dto';
import { ServiceDetail } from 'src/service-detail/service-detail.entity';
import { ServiceDetailImage } from 'src/service-detail-image/service-detail-image.entity';
import { ServiceBullet } from 'src/service-detail-point/service-detail-point.entity';
import { pageHeadings, PageType } from './page-heading.entity';

@Injectable()
export class ServiceService {
  constructor(
    @InjectRepository(Service)
    private readonly serviceRepo: Repository<Service>,

    @InjectRepository(ServiceDetail)
    private readonly serviceDetailRepo: Repository<ServiceDetail>,

    @InjectRepository(ServiceDetailImage)
    private readonly serviceDetailImageRepo: Repository<ServiceDetailImage>,

    @InjectRepository(pageHeadings)
    private readonly pageHeading: Repository<pageHeadings>,

    private readonly dataSource: DataSource,
  ) {}

  async create(dto: any, createdBy: number, service_image: any) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    // try {
    const existingService = await queryRunner.manager.findOne(Service, {
      where: { slug: dto.slug },
    });

    if (existingService) {
      // Option 1: Throw a readable error
      throw new Error(
        `Slug "${dto.slug}" already exists. Please use another slug.`,
      );

      // Option 2: Auto-generate a unique one instead
      // dto.slug = `${dto.slug}-${Date.now()}`;
    }
    const service = queryRunner.manager.create(Service, {
      heading: dto.heading,
      title: dto.title,
      description: dto.description,
      banner_heading: dto.banner_heading,
      banner_description: dto.banner_description,
      slug: dto.slug,
      meta_title: dto.meta_title,
      service_image_link: 'services/' + service_image,
      meta_description: dto.meta_description,
      schema_json: dto.schema_json,
      canonical_url: dto.canonical_url,
      keyword: dto.keyword,
      created_by: createdBy,
    });

    const savedService = await queryRunner.manager.save(service);

    let savedDetails: any[] = [];

    if (Array.isArray(dto.details) && dto.details.length > 0) {
      const detailsToSave = dto.details.map((detail) =>
        queryRunner.manager.create(ServiceDetail, {
          service_id: savedService.id,
          heading: detail.heading || null,
          title: detail.title,
          description: detail.description,
          type: detail.type || null,
          button_text: detail.button_text || null,
          button_Url: detail.button_Url || null,
          image: detail.image || null,
          alter_image: detail.alter_image || null,
          image_name: detail.image_name || null,
          created_by: createdBy,
        }),
      );

      savedDetails = await queryRunner.manager.save(detailsToSave);
    }

    // Step 3: Save all images linked to each detail
    const allImageRecords: ServiceDetailImage[] = [];

    if (Array.isArray(dto.details) && Array.isArray(savedDetails)) {
      for (let i = 0; i < dto.details.length; i++) {
        const detailDto = dto.details[i];
        const savedDetail = savedDetails[i];

        // Skip if no matching saved detail
        if (!savedDetail) continue;

        // Handle images for this detail
        if (Array.isArray(detailDto?.images) && detailDto.images.length > 0) {
          const imageEntities = detailDto.images.map((img) => {
            // 🧹 Clean filename: remove fieldname like "imageGroups[0][]-" prefix
            const cleanFilename = img.image_path
              ?.replace(/^.*?-/, '') // remove everything before the first dash
              ?.trim();

            return queryRunner.manager.create(ServiceDetailImage, {
              service_detail_id: savedDetail.id,
              image_path: `services/${cleanFilename}`,
              status: img.status ?? 1, // default active
            });
          });

          allImageRecords.push(...imageEntities);
        }
      }
    }

    if (allImageRecords.length > 0) {
      await queryRunner.manager.save(allImageRecords);
    }

    const allBulletRecords: ServiceBullet[] = [];

    if (Array.isArray(dto.details)) {
      for (let i = 0; i < dto.details.length; i++) {
        const detailDto = dto.details[i];
        const savedDetail = savedDetails[i];

        if (detailDto?.bullets && detailDto.bullets.length > 0) {
          const bulletEntities = detailDto.bullets.map((b) =>
            queryRunner.manager.create(ServiceBullet, {
              service_detail_id: savedDetail.id,
              title: b.title,
              text: b.text || null,
              icon: b.icon || null,
              alt_icon: b.alt_icon,
              status: b.status ?? 1,
            }),
          );
          allBulletRecords.push(...bulletEntities);
        }
      }
    }

    if (allBulletRecords.length > 0) {
      await queryRunner.manager.save(allBulletRecords);
    }

    await queryRunner.commitTransaction();

    return {
      message: 'Service and details inserted successfully',
      service: savedService,
      details: savedDetails || [],
      images: allImageRecords,
      bullets: allBulletRecords,
    };
    // } catch (error) {
    //     await queryRunner.rollbackTransaction();
    //     console.error('Insert error:', error);
    //     return { message: `Failed to insert service and details: ${error.message}` };
    // }
    // finally {
    //     await queryRunner.release();
    // }
  }
  async findAll(status?: number, limit: number = 10, offset: number = 0) {
    try {
      const typer = PageType.SERVICE;
      const page_heading = await this.pageHeading.findOne({
        where: { type: typer },
      });

      // ✅ Build where condition
      const whereCondition: any = { status: Not(2) };
      if (status !== undefined) {
        whereCondition.status = status ?? 1;
      }

      // ✅ Base query
      const queryOptions: any = {
        where: whereCondition,
        relations: ['details', 'details.images', 'details.bullets'],
        order: { id: 'DESC' },
        skip: offset,
        take: limit,
      };

      // ✅ Apply limit if provided by frontend
      if (limit && !isNaN(limit)) {
        queryOptions.take = limit;
      }

      const [services, tottal] =
        await this.serviceRepo.findAndCount(queryOptions);

      return {
        status: true,
        message: 'Services fetched successfully',
        page_heading,
        total: tottal,
        data: services,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch services',
        data: error.message,
      };
    }
  }

  // 🔍 FIND BY ID with join
  async findById(slug: string): Promise<any> {
    try {
      const service = await this.serviceRepo.findOne({
        where: { slug: slug, status: Not(2) },
        relations: ['details', 'details.images', 'details.bullets'],
      });

      if (!service) {
        return {
          status: false,
          message: `Service with Slug ${slug} not found`,
          data: null,
        };
      }

      return {
        status: true,
        message: 'Service fetched successfully',
        data: service,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch service',
        data: error.message,
      };
    }
  }

  async update(
    id: number,
    dto: any,
    updatedBy: number,
    serviceImageFile?: string,
  ) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      // Step 1: Find existing service
      const existingService = await queryRunner.manager.findOne(Service, {
        where: { id },
      });
      if (!existingService) {
        return { message: 'Service not found' };
      }

      // Step 2: Update main service fields
      const updatableFields = [
        'heading',
        'title',
        'description',
        'banner_heading',
        'banner_description',
        'slug',
        'meta_title',
        'meta_description',
        'schema_json',
        'canonical_url',
        'keyword',
      ];

      for (const field of updatableFields) {
        if (dto[field] !== undefined) {
          existingService[field] = dto[field];
        }
      }

      if (serviceImageFile) {
        existingService.service_image_link = `services/${serviceImageFile}`;
      }

      await queryRunner.manager.save(existingService);

      // Step 3: Existing details
      const existingDetails = await queryRunner.manager.find(ServiceDetail, {
        where: { service_id: id },
        relations: ['images', 'bullets'],
      });

      // Step 4: Handle incoming details
      if (Array.isArray(dto.details) && dto.details.length > 0) {
        for (const detailDto of dto.details) {
          let detailEntity: ServiceDetail;

          const detailId = detailDto.id ? Number(detailDto.id) : null;

          // ✅ Update existing detail
          if (detailId) {
            const foundDetail = existingDetails.find((d) => d.id === detailId);
            if (!foundDetail) continue;
            detailEntity = foundDetail;

            const detailFields = [
              'heading',
              'title',
              'description',
              'type',
              'button_text',
              'button_Url',
              'image',
              'alter_image',
              'image_name',
            ];

            for (const field of detailFields) {
              if (detailDto[field] !== undefined) {
                (detailEntity as any)[field] = detailDto[field];
              }
            }

            await queryRunner.manager.save(detailEntity);
          }
          // ✅ Create new detail
          else {
            detailEntity = queryRunner.manager.create(ServiceDetail, {
              service_id: id,
              heading: detailDto.heading,
              title: detailDto.title,
              description: detailDto.description,
              type: detailDto.type,
              button_text: detailDto.button_text,
              button_Url: detailDto.button_Url,
              image: detailDto.image,
              alter_image: detailDto.alter_image,
              image_name: detailDto.image_name,
            });
            await queryRunner.manager.save(detailEntity);
          }

          // Step 5: Handle images
          if (Array.isArray(detailDto.images) && detailDto.images.length > 0) {
            for (const imgDto of detailDto.images) {
              const imageId = imgDto.id ? Number(imgDto.id) : null;

              if (imageId) {
                // 🔹 Update existing image
                const existingImage = await queryRunner.manager.findOne(
                  ServiceDetailImage,
                  {
                    where: {
                      id: imageId,
                      service_detail_id: detailEntity.id,
                    },
                  },
                );
                if (!existingImage) continue;

                if (imgDto.image_path !== undefined)
                  existingImage.image_path = `services/${imgDto.image_path}`;
                if (imgDto.status !== undefined)
                  existingImage.status = imgDto.status;

                await queryRunner.manager.save(existingImage);
              } else {
                // 🔹 Insert new image
                const newImage = queryRunner.manager.create(
                  ServiceDetailImage,
                  {
                    service_detail_id: detailEntity.id,
                    image_path: `services/${imgDto.image_path}`,
                    status: imgDto.status ?? 1,
                  },
                );
                await queryRunner.manager.save(newImage);
              }
            }
          }

          // Step 6: Handle bullets
          if (
            Array.isArray(detailDto.bullets) &&
            detailDto.bullets.length > 0
          ) {
            for (const bulletDto of detailDto.bullets) {
              const bulletId = bulletDto.id ? Number(bulletDto.id) : null;

              if (bulletId) {
                // 🔹 Update existing bullet
                const existingBullet = await queryRunner.manager.findOne(
                  ServiceBullet,
                  {
                    where: {
                      id: bulletId,
                      service_detail_id: detailEntity.id,
                    },
                  },
                );
                if (!existingBullet) continue;

                const bulletFields = [
                  'title',
                  'text',
                  'icon',
                  'alt_icon',
                  'status',
                ];
                for (const field of bulletFields) {
                  if (bulletDto[field] !== undefined) {
                    (existingBullet as any)[field] = bulletDto[field];
                  }
                }

                await queryRunner.manager.save(existingBullet);
              } else {
                // 🔹 Create new bullet
                const newBullet = queryRunner.manager.create(ServiceBullet, {
                  service_detail_id: detailEntity.id,
                  title: bulletDto.title,
                  text: bulletDto.text,
                  icon: bulletDto.icon,
                  alt_icon: bulletDto.alt_icon,
                  status: bulletDto.status ?? 1,
                });
                await queryRunner.manager.save(newBullet);
              }
            }
          }
        }
      }

      await queryRunner.commitTransaction();

      // Step 7: Return updated service with relations
      const updatedService = await this.dataSource
        .getRepository(Service)
        .findOne({
          where: { id },
          relations: ['details', 'details.images', 'details.bullets'],
        });

      return {
        message: 'Service and details updated successfully',
        service: updatedService,
      };
    } catch (error) {
      await queryRunner.rollbackTransaction();
      console.error('Update error:', error);
      return { message: `Failed to update service: ${error.message}` };
    } finally {
      await queryRunner.release();
    }
  }

  // 🔁 TOGGLE STATUS
  async toggleStatus(id: number) {
    try {
      const service = await this.serviceRepo.findOne({
        where: { id },
        relations: ['details'],
      });

      if (!service) {
        return {
          status: false,
          message: `Service with ID ${id} not found`,
          data: null,
        };
      }

      // Toggle main service status
      service.status = service.status === 1 ? 0 : 1;
      await this.serviceRepo.save(service);

      // Sync details’ statuses
      await this.serviceDetailRepo
        .createQueryBuilder()
        .update(ServiceDetail)
        .set({ status: service.status })
        .where('service_id = :id', { id })
        .execute();

      return {
        status: true,
        message: `Service and details ${service.status === 1 ? 'activated' : 'deactivated'} successfully`,
        data: service,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to toggle service status',
        data: error.message,
      };
    }
  }

  async createPageHeading(body: any) {
    try {
      const typer =
        body.type == 'program' ? PageType.PROGRAM : PageType.SERVICE;

      const exist = await this.pageHeading.findOne({ where: { type: typer } });
      if (exist) {
        return { status: false, message: 'Already Exist' };
      }
      const newHeading = this.pageHeading.create({
        ...body,
        type: typer,
      });
      const result = await this.pageHeading.save(newHeading);
      return {
        status: true,
        message: 'Page heading created successfully!',
        data: result,
      };
    } catch (error) {
      return { status: false, message: error.message };
    }
  }

  async findPageOne(type: string) {
    try {
      const typer = type == 'program' ? PageType.PROGRAM : PageType.SERVICE;

      console.log(typer);
      const heading = await this.pageHeading.findOne({
        where: { type: typer },
      });
      if (!heading) {
        return { status: false, message: 'Page heading not found' };
      }
      return {
        status: true,
        message: 'Page heading fetched successfully!',
        data: heading,
      };
    } catch (error) {
      return { status: false, message: error.message };
    }
  }

  async updatePage(id: number, body: any) {
    try {
      const heading = await this.pageHeading.findOne({ where: { id } });
      if (!heading) {
        return { status: false, message: 'Page heading not found' };
      }

      Object.assign(heading, body);
      const updated = await this.pageHeading.save(heading);

      return {
        status: true,
        message: 'Page heading updated successfully!',
        data: updated,
      };
    } catch (error) {
      return { status: false, message: error.message };
    }
  }

  async remove(id: number) {
    try {
      const result = await this.pageHeading.delete(id);
      if (result.affected === 0) {
        return { status: false, message: 'Page heading not found' };
      }
      return { status: true, message: 'Page heading deleted successfully!' };
    } catch (error) {
      return { status: false, message: error.message };
    }
  }

  async softDelete(id: number) {
    try {
      const service = await this.serviceRepo.findOne({
        where: { id },
        relations: ['details'],
      });

      if (!service) {
        return {
          status: false,
          message: `Service with ID ${id} not found`,
          data: null,
        };
      }

      // If already soft deleted
      if (service.status === 2) {
        return {
          status: false,
          message: `Service with ID ${id} is already deleted`,
          data: null,
        };
      }

      // Generate deleted slug
      const baseSlug = `${service.slug}-DL`;

      // Check for existing deleted slugs
      const existingDeleted = await this.serviceRepo
        .createQueryBuilder('s')
        .where('s.slug LIKE :slug', { slug: `${baseSlug}%` })
        .getMany();

      // Determine suffix number if duplicate slug found
      let newSlug = baseSlug;
      if (existingDeleted.length > 0) {
        newSlug = `${baseSlug}${existingDeleted.length}`;
      }

      // Update service status & slug
      service.status = 2;
      service.slug = newSlug;
      await this.serviceRepo.save(service);

      // Soft delete all related service details
      await this.serviceDetailRepo
        .createQueryBuilder()
        .update(ServiceDetail)
        .set({ status: 2 })
        .where('service_id = :id', { id })
        .execute();

      return {
        status: true,
        message: `Service soft deleted successfully with slug: ${newSlug}`,
        data: service,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to soft delete service',
        data: error.message,
      };
    }
  }
}
