import { Injectable, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Transaction } from './transaction.entity';
import { CreateTransactionDto } from './dto/create-transaction.dto';
import { Category } from 'src/category/category.entity';

@Injectable()
export class TransactionService {
  constructor(
    @InjectRepository(Transaction)
    private readonly transactionRepo: Repository<Transaction>,

    @InjectRepository(Category)
    private readonly categoryRepo: Repository<Category>,
  ) {}

  async create(dto: CreateTransactionDto) {
    try {
      const category = dto.category_id
        ? await this.categoryRepo.findOne({ where: { id: dto.category_id } })
        : undefined;
      if (!category && dto.category_id) {
        console.log(`Category with id ${dto.category_id} not found.`);
        throw new NotFoundException("Category not found");
      } 
      const transaction = this.transactionRepo.create({
        ...dto,
        category: category as Category,
      });
      const saved = await this.transactionRepo.save(transaction);

      return {
        status: true,
        message: 'Transaction created successfully',
        data: saved,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to create transaction',
        data: error.message,
      };
    }
  }

  async findAll() {
    try {
      const transactions = await this.transactionRepo
        .createQueryBuilder('transaction')
        .leftJoinAndSelect('transaction.category', 'category')
        .leftJoin(
          'programs',
          'program',
          'program.id = transaction.support_program',
        )
        .leftJoin(
          'services',
          'service',
          'service.id = transaction.support_service',
        )
        .orderBy('transaction.id', 'DESC')
        .select([
          'transaction.id AS id',
          'transaction.transaction_id AS transaction_id',
          'transaction.status AS status',
          'transaction.amount AS amount',
          'transaction.currency_code AS currency_code',
          'transaction.payment_through AS payment_through',
          'transaction.invoice_no AS invoice_no',
          'transaction.name AS name',
          'transaction.email AS email',
          'transaction.address AS address',
          'transaction.telephone AS telephone',
          'transaction.support_program AS support_program',
          'transaction.support_service AS support_service',
          'transaction.created_at AS created_at',

          // Related fields
          'category.id AS category_id',
          'category.name AS category_name',
          'program.id AS program_id',
          'program.title AS program_title',
          'service.id AS service_id',
          'service.title AS service_title',
        ])
        .getRawMany();

      // Format response neatly
      const formattedTransactions = transactions.map((t) => ({
        id: t.id,
        transaction_id: t.transaction_id,
        status: t.status,
        amount: t.amount,
        currency_code: t.currency_code,
        payment_through: t.payment_through,
        invoice_no: t.invoice_no,
        name: t.name,
        email: t.email,
        address: t.address,
        telephone: t.telephone,
        created_at: t.created_at,

        // these store ids
        support_program: t.support_program,
        support_service: t.support_service,

        // show relations (titles + ids)
        category: t.category_id
          ? { category_id: t.category_id, category_name: t.category_name }
          : null,

        program: t.program_id
          ? { program_id: t.program_id, program_title: t.program_title }
          : null,

        service: t.service_id
          ? { service_id: t.service_id, service_title: t.service_title }
          : null,
      }));

      return {
        status: true,
        message: 'Transactions fetched successfully',
        data: formattedTransactions,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch transactions',
        data: error.message,
      };
    }
  }

  async findByProgramOrService(programId?: number, serviceId?: number) {
    try {
      const baseQuery = this.transactionRepo
        .createQueryBuilder('transaction')
        .leftJoinAndSelect('transaction.category', 'category')
        .leftJoin(
          'programs',
          'program',
          'program.id = transaction.support_program',
        )
        .leftJoin(
          'services',
          'service',
          'service.id = transaction.support_service',
        )
        .orderBy('transaction.id', 'DESC')
        .select([
          'transaction.id AS id',
          'transaction.transaction_id AS transaction_id',
          'transaction.status AS status',
          'transaction.amount AS amount',
          'transaction.currency_code AS currency_code',
          'transaction.payment_through AS payment_through',
          'transaction.invoice_no AS invoice_no',
          'transaction.name AS name',
          'transaction.email AS email',
          'transaction.address AS address',
          'transaction.telephone AS telephone',
          'transaction.support_program AS support_program',
          'transaction.support_service AS support_service',
          'transaction.created_at AS created_at',

          // Related data
          'category.id AS category_id',
          'category.name AS category_name',
          'program.id AS program_id',
          'program.title AS program_title',
          'service.id AS service_id',
          'service.title AS service_title',
        ]);

      // ✅ Apply dynamic filters
      if (programId) {
        baseQuery.andWhere('transaction.support_program = :programId', {
          programId,
        });
      }

      if (serviceId) {
        baseQuery.andWhere('transaction.support_service = :serviceId', {
          serviceId,
        });
      }

      // ✅ Fetch transactions
      const transactions = await baseQuery.getRawMany();

      if (!transactions.length) {
        return {
          status: false,
          message: 'No transactions found for given filters',
          data: [],
        };
      }

      // ✅ Format transactions
      const formattedTransactions = transactions.map((t) => ({
        id: t.id,
        transaction_id: t.transaction_id,
        status: t.status,
        amount: parseFloat(t.amount),
        currency_code: t.currency_code,
        payment_through: t.payment_through,
        invoice_no: t.invoice_no,
        name: t.name,
        email: t.email,
        address: t.address,
        telephone: t.telephone,
        created_at: t.created_at,
        support_program: t.support_program,
        support_service: t.support_service,

        category: t.category_id
          ? { category_id: t.category_id, category_name: t.category_name }
          : null,

        program: t.program_id
          ? { program_id: t.program_id, program_title: t.program_title }
          : null,

        service: t.service_id
          ? { service_id: t.service_id, service_title: t.service_title }
          : null,
      }));

      // ✅ Get total count and total amount
      const summaryQuery = this.transactionRepo
        .createQueryBuilder('transaction')
        .select('COUNT(transaction.id)', 'total_transactions')
        .addSelect('COALESCE(SUM(transaction.amount), 0)', 'total_amount');

      if (programId) {
        summaryQuery.andWhere('transaction.support_program = :programId', {
          programId,
        });
      }

      if (serviceId) {
        summaryQuery.andWhere('transaction.support_service = :serviceId', {
          serviceId,
        });
      }

      const summary = await summaryQuery.getRawOne();

      return {
        status: true,
        message: 'Filtered transactions fetched successfully',
        total_transactions: parseInt(summary.total_transactions, 10),
        total_amount: parseFloat(summary.total_amount),
        data: formattedTransactions,
      };
    } catch (error) {
      return {
        status: false,
        message: 'Failed to fetch filtered transactions',
        data: error.message,
      };
    }
  }
}
