import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Video } from './video.entity';
import { CreateVideoDto } from './dto/create-video.dto';
import { UpdateVideoDto } from './dto/update-video.dto';

@Injectable()
export class VideoService {
    constructor(
        @InjectRepository(Video)
        private readonly videoRepo: Repository<Video>,
    ) {}

    async create(dto: CreateVideoDto) {
        try {
            const video = this.videoRepo.create({
                ...dto,
                type: 'embedded',
            });
            const saved = await this.videoRepo.save(video);

            return {
                status: true,
                message: 'Video created successfully',
                data: saved,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to create video',
                data: null,
            };
        }
    }

    async findAll() {
        try {
            const videos = await this.videoRepo.find({
                order: { id: 'DESC' },
            });

            return {
                status: true,
                message: 'Videos fetched successfully',
                data: videos,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to fetch videos',
                data: null,
            };
        }
    }

    async findOne(id: number) {
        try {
            const video = await this.videoRepo.findOne({ where: { id } });

            if (!video) {
                return {
                    status: false,
                    message: `Video not found with ID ${id}`,
                    data: null,
                };
            }

            return {
                status: true,
                message: 'Video fetched successfully',
                data: video,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to fetch video',
                data: null,
            };
        }
    }

    async update(id: number, dto: UpdateVideoDto) {
        try {
            const existing = await this.videoRepo.findOne({ where: { id } });
            if (!existing) {
                return {
                    status: false,
                    message: `Video not found with ID ${id}`,
                    data: null,
                };
            }

            Object.assign(existing, dto);
            const updated = await this.videoRepo.save(existing);

            return {
                status: true,
                message: 'Video updated successfully',
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to update video',
                data: null,
            };
        }
    }

    async toggleStatus(id: number) {
        try {
            const video = await this.videoRepo.findOne({ where: { id } });
            if (!video) {
                return {
                    status: false,
                    message: `Video not found with ID ${id}`,
                    data: null,
                };
            }

            video.status = video.status === 1 ? 0 : 1;
            const updated = await this.videoRepo.save(video);

            return {
                status: true,
                message: `Video status toggled to ${updated.status === 1 ? 'Active' : 'Inactive'}`,
                data: updated,
            };
        } catch (error) {
            return {
                status: false,
                message: error.message || 'Failed to toggle video status',
                data: null,
            };
        }
    }
}
